(function (RongIM, dependencies, components) {
'use strict';

var common = RongIM.common;
var utils = RongIM.utils;
var $ = dependencies.jQuery;
var pageNum = RongIM.config.profile.pageNum;

/*
说明： 选择星标联系人通用组件
    props
        selected            选中的用户
        defaultSelected     默认选中的用户
        canNotSelected      不可选择的用户
        disableExecutive    禁用高管模式
    data
        keywork             搜索关键字
        searchResult        页面绑定搜索结果
        members             所有的星标联系人
*/
function getStar(resolve, reject) {
    var im = RongIM.instance;
    var dataModel = im.dataModel;
    var starApi = dataModel.Star;
    var userApi = im.dataModel.User;
    var options = {
        name: 'star',
        template: 'modules/star/templates/group-star.html',
        data: function () {
            return {
                keyword: '',
                searchResult: [],
                members: [],
                pageNum: pageNum,
                currentPage: 1
            };
        },
        props: ['selected', 'defaultSelected', 'canNotSelected', 'disableExecutive'],
        computed: {
            defaultIdList: function () {
                return (this.defaultSelected || []).map(function (item) {
                    return item.id;
                });
            },
            pageList: function () {
                var _list = this.searchResult;
                var end = this.currentPage * this.pageNum;
                return _list.slice(0, end);
            },
            canNotSelectedIdList: function () {
                return (this.canNotSelected || []).map(function (item) {
                    return item.id;
                });
            },
            /* 页面绑定 "全选按钮" */
            checkedAll: {
                get: function () {
                    return getCheckedAll(this);
                },
                set: function (value) {
                    setCheckedAll(this, value);
                }
            },
            /* "全选按钮" 的半选状态 */
            indeterminate: function () {
                if (this.isCardSelect) {
                    return false;
                }
                return (typeof this.checkedAll) !== 'boolean';
            },
            /* 用户的选中状态 */
            checked: {
                get: function () {
                    return this.selected.map(function (item) {
                        return item.id;
                    });
                },
                set: function (newMemberIds) {
                    var context = this;
                    userApi.get(newMemberIds, function (errorCode, newMembers) {
                        newMembers = [].concat(newMembers);
                        if (errorCode) {
                            context.toastError(errorCode);
                            return;
                        }
                        userApi.get(context.checked, function (UerrorCode, oldMembers) {
                            oldMembers = [].concat(oldMembers);
                            if (UerrorCode) {
                                context.toastError(UerrorCode);
                                return;
                            }
                            var addedList = common.without(newMembers, oldMembers);
                            if (addedList.length > 0) context.$emit('added', addedList);

                            var removedList = common.without(oldMembers, newMembers);
                            var listLen = removedList.length;
                            if (listLen > 0) context.$emit('removed', removedList);
                        });
                    });
                    if (context.keyword) context.clear();
                }
            },
            isCardSelect: function () {
                return this.$parent.$options.name === 'card';
            }
        },
        components: {
            avatar: components.getAvatar
        },
        watch: {
            keyword: function (keyword) {
                keywordChanged(this, keyword);
            }
        },
        created: function () {
            created(this, starApi);
        },
        methods: {
            toastError: function (errorCode) {
                var el = null;
                if (this.$parent) {
                    el = this.$parent.$el.firstChild;
                }
                common.toastError(errorCode, el);
            },
            getUsername: function (item) {
                var name = common.getUsernameHighlight(item);
                return common.filterMark(name);
            },
            executiveLimit: function (item) {
                if ((item && item.isFriend) || im.auth.isExecutive || this.disableExecutive) {
                    return false;
                }
                var isExecutive = item && !!item.isExecutive;
                return isExecutive;
            },
            isDefault: function (item) {
                return this.defaultIdList.indexOf(item.id) >= 0;
            },
            showNotSelect: function (item) {
                var canNotSelected = item && this.canNotSelectedIdList.indexOf(item.id) > -1;
                return this.executiveLimit(item) || canNotSelected;
            },
            isDisabled: function (item) {
                var canNotSelected = this.canNotSelectedIdList.indexOf(item.id) > -1;
                return this.isDefault(item) || this.executiveLimit(item) || canNotSelected;
            },
            clear: function () {
                this.keyword = '';
            },
            loadMore: function () {
                loadMore(this);
            }
        }
    };
    utils.asyncComponent(options, resolve, reject);
}

/*
说明： 判断 "全选按钮" 的全选状态
*/
function getCheckedAll(context) {
    // 当前搜索结果中选中的用户的列表
    var starChecked = [];
    var checkedIdList = context.checked;
    context.searchResult.forEach(function (item) {
        var existed = item && checkedIdList.indexOf(item.id) >= 0;
        if (existed) starChecked.push(item);
    });
    var length = starChecked.length;
    var result;
    if (length > 0) {
        // 有效的用户 排除高管和不可选中的用户
        var validMemberList = context.searchResult.filter(function (item) {
            var nothaveExecutivelimit = !context.executiveLimit(item);
            var canSelected = context.canNotSelectedIdList.indexOf(item.id) === -1;
            return nothaveExecutivelimit && canSelected;
        });
        var isAll = length === validMemberList.length;
        result = isAll ? true : null;
    } else {
        result = false;
    }
    return result;
}

/*
说明： 全选当前搜索展示的用户
*/
function setCheckedAll(context, value) {
    var members = context.searchResult;
    // 有效的用户 id 排除高管和不可选中的用户
    var validMemberIdList = members.filter(function (item) {
        var nothaveExecutivelimit = !context.executiveLimit(item);
        var canSelected = context.canNotSelectedIdList.indexOf(item.id) === -1;
        return nothaveExecutivelimit && canSelected;
    }).map(function (item) {
        return item.id;
    });
    if (value) {
        context.checked = [].concat(validMemberIdList, context.checked);
    } else {
        context.checked = context.checked.filter(function (id) {
            return validMemberIdList.indexOf(id) < 0;
        });
    }
}

function keywordChanged(context, keyword) {
    if (keyword.length === 0) {
        if (context.searchResult.length === 0) {
            context.searchResult = context.members;
        }
    } else {
        context.searchResult = [];
        var searchResult = [];
        context.members.forEach(function (item) {
            var nameRange = utils.searchStrRange(item.name, context.keyword);
            var aliasRange = utils.searchStrRange(item.alias, context.keyword);
            if (nameRange || aliasRange) {
                var result = $.extend({
                    range: nameRange,
                    aliasRange: aliasRange
                }, item);
                searchResult.push(result);
            }
        });
        common.sortUsers(searchResult);
        context.searchResult = searchResult;
    }
}

function created(context, starApi) {
    starApi.getList(function (errorCode, list) {
        if (errorCode) {
            context.toastError(errorCode);
            return;
        }
        common.sortUsers(list);
        context.members = list;
        context.searchResult = list;
    });
}

function loadMore(context) {
    var end = context.currentPage * context.pageNum;
    var list = context.searchResult;
    if (list && list.length > end) {
        context.currentPage += 1;
    }
}

$.extend(true, components, {
    group: {
        getStar: getStar
    }
});
}(RongIM, {
    jQuery: jQuery
}, RongIM.components));
